<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
require_once __DIR__ . '/BotInterface.php';

class ChatGPTBot implements BotInterface {

    public static function getKey() {
        return 'chatgpt';
    }
    
    public static function getName() {
        return 'ChatGPT';
    }
    
    public static function getTemplate() {

        $models = array(
            "gpt-3.5-turbo"=>"gpt-3.5-turbo",
            "gpt-3.5-turbo-0125"=>"gpt-3.5-turbo-0125",
            "gpt-3.5-turbo-1106"=>"gpt-3.5-turbo-1106",
            "gpt-3.5-turbo-16k"=>"gpt-3.5-turbo-16k",
            "gpt-4"=>"gpt-4",
            "gpt-4-0125-preview"=>"gpt-4-0125-preview",
            "gpt-4-0613"=>"gpt-4-0613",
            "gpt-4.1"=>"gpt-4.1",
            "gpt-4-1106-preview"=>"gpt-4-1106-preview",
            "gpt-4.1-2025-04-14"=>"gpt-4.1-2025-04-14",
            "gpt-4.1-mini"=>"gpt-4.1-mini",
            "gpt-4.1-mini-2025-04-14"=>"gpt-4.1-mini-2025-04-14",
            "gpt-4.1-nano"=>"gpt-4.1-nano",
            "gpt-4.1-nano-2025-04-14"=>"gpt-4.1-nano-2025-04-14",
            "gpt-4.5-preview"=>"gpt-4.5-preview",
            "gpt-4.5-preview-2025-02-27"=>"gpt-4.5-preview-2025-02-27",
            "gpt-4o"=>"gpt-4o",
            "gpt-4o-2024-05-13"=>"gpt-4o-2024-05-13",
            "gpt-4o-2024-08-06"=>"gpt-4o-2024-08-06",
            "gpt-4o-2024-11-20"=>"gpt-4o-2024-11-20",
            "gpt-4o-mini"=>"gpt-4o-mini",
            "gpt-4o-mini-2024-07-18"=>"gpt-4o-mini-2024-07-18",
            "gpt-4-turbo"=>"gpt-4-turbo",
            "gpt-4-turbo-2024-04-09"=>"gpt-4-turbo-2024-04-09",
            "gpt-4-turbo-preview"=>"gpt-4-turbo-preview",
            "gpt-5"=>"gpt-5",
            "gpt-5-mini-2025-08-07"=>"gpt-5-mini-2025-08-07",
            "gpt-5-mini"=>"gpt-5-mini",
            "gpt-5-nano-2025-08-07"=>"gpt-5-nano-2025-08-07",
            "o1"=>"o1",
            "o1-2024-12-17"=>"o1-2024-12-17",
            "o1-mini"=>"o1-mini",
            "o1-mini-2024-09-12"=>"o1-mini-2024-09-12",
            "o1-preview"=>"o1-preview",
            "o1-preview-2024-09-12"=>"o1-preview-2024-09-12",
            "o1-pro"=>"o1-pro",
            "o1-pro-2025-03-19"=>"o1-pro-2025-03-19",
            "o3-mini"=>"o3-mini",
            "o3-mini-2025-01-31"=>"o3-mini-2025-01-31",
            "o4-mini"=>"o4-mini",
            "o4-mini-2025-04-16"=>"o4-mini-2025-04-16",
            "chatgpt-4o-latest"=>"chatgpt-4o-latest"
        );

        return array(
            "openai_token" => array("type" => "text", "default" => ""),
            "model"             => array( "type" => "select",   "default"=>"gpt-3.5-turbo", "options"=>$models),
            "system_prompt"     => array( "type" => "textarea", "default"=>"You are a helpful assistant"),
            "action_prompt"     => array( "type" => "textarea", "default"=>"If the user is frustrated or angry, or if the user asks to connect to a real person, you will pass the conversation to a human by appending '[transfer]' to your reply. In the farewell/goodbye message you will append '[close]' to your reply."),
            "guardrails_prompt" => array( "type" => "textarea", "default"=>". Do not answer anything that is not in your system prompt context."),
            "transcribe_audio"  => array( "type" => "select",   "default"=>"no", "options"=>array("yes"=>__('Yes'),"no"=>__('No'))),
            "voice_reply"       => array( "type" => "select",   "default"=>"no", "options"=>array("yes"=>__('Yes'),"no"=>__('No'))),
            "debug"             => array( "type" => "select",   "default"=>"no", "options"=>array("yes"=>__('Yes'),"no"=>__('No')))
        );
    }
}

if (!defined('BOT_REGISTRATION_MODE') || !BOT_REGISTRATION_MODE) {
    require_once __DIR__ . '/common_functions.php';
    require_once(__DIR__."/../chatbroker.class.".PHP_MAJOR_VERSION.".php");

    $logfile = basename(__FILE__,'.php');
    $iniFile = isset($_REQUEST['bot']) ? getcwd()."/".$_REQUEST['bot'].".ini" : null;
    $config = BotConfig::load('ChatGPTBot', $iniFile);

    if ($config === null) {
        header('HTTP/1.0 403 Forbidden');
        die("Configuration file missing or inaccessible");
    }

    // Get bot input DATA
    $json = file_get_contents('php://input');
    $data = json_decode($json,1);

    if(!isset($data)) { 
        header('HTTP/1.0 403 Forbidden');
        die("Bad call");
    }
    bot_debug($logfile,"POST RECEIVED:\n".print_r($data,1));
    bot_debug($logfile,"CONFIG:\n".print_r($config,1));

    $cb = new ChatBroker(array());

    // process audio attachments and transcribe using openai
    if(isset($data['attach']['mime']) && $config['transcribe_audio']=='yes') {
        $audio_file = preg_replace("/; codecs=opus/","",$data['attach']['file']);
        $mime = preg_replace("/; codecs=opus/","",$data['attach']['mime']);
        if(preg_match("/audio/",$mime)) {
            $files = $cb->check_for_attachments($data['msgid']);
            foreach($files as $file) {
                // retrieve file to temp disk to get metadata (mime type, image info, etc)
                $path = $cb->chatbroker_host ."/api/files/messages/".$data['msgid']."/".$file;
                $type = pathinfo($path, PATHINFO_EXTENSION);
                $filedata = file_get_contents($path);
                file_put_contents("/tmp/$file",$filedata);
                $text  = openai_whisper("/tmp/$file",$config['openai_token']);
                if($text!='') $data['body']=$text;
            }
        }
    }

    $user_prompt = $data['body'];

    $user_prompt.=$config['guardrails_prompt']; // add guardrails to question

    $chat_context = array_slice($data['chat_context'],-5);
    $chat_context[] = array("role"=>"user","content"=>$user_prompt);

    bot_debug($logfile,"CONTEXT:\n".print_r($chat_context,1));

    if($config['openai_token']=='') {
        $response = "Bot is not yet configured, be sure to set up your OpenAI API key";
    } else {
        $response = ask_openai($config['openai_token'],$config['system_prompt']." ".$config['action_prompt'],$chat_context);
    }

    if($config['voice_reply']=='no') {
        $return = array("body"=>$response,"attach"=>'');
    } else {

        list($command, $parameter, $msg) = $cb->extractCommandFromBotReply($response); // do not send [commands] to TTS
        $attach = openai_tts($msg,"alloy",$config['openai_token']);

        if(is_null($attach)) {
            $return = array("body"=>$response,"attach"=>'');
        } else {
            if($msg!='') {
                // Do not send audio if there is no actual audio to send (avoid empty audio responses)
                $return = array("body"=>$response,"attach"=>$attach);
            } else {
                $return = array("body"=>$response,"attach"=>'');
            }
        }
    }

    bot_debug($logfile,"REPLY: $response");
    echo json_encode($return);
    die();
}

function ask_openai($openai_token,$system_prompt,$user_prompt) {
    global $config, $logfile;

    bot_debug($logfile,"ask openai $openai_token, $system_prompt, $user_prompt\n");
    $url = 'https://api.openai.com/v1/chat/completions';
    $sysprompt = array();
    $sysprompt[]=array("role"=>"system","content"=>$system_prompt);
    $messages = array_merge($sysprompt,$user_prompt);
    $data = array(
       "model"=> $config['model'],
       "messages" => $messages
    );
    $data_string = json_encode($data);
    bot_debug($logfile,"OpenAI Messages: ".print_r($messages,1));
    $headers=array('Content-Type: application/json','Content-Length: ' . strlen($data_string),"Authorization: Bearer $openai_token");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 120);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"OpenAI response: ".print_r($response,1));
    if(isset($response['choices'][0]['message']['content'])) {
        return $response['choices'][0]['message']['content'];
    } else if(isset($response['error'])) {
        return $response['error']['message'];
    } else {
        return "02 - Digital Brain is not working";
    }
}

function openai_whisper($file,$token) {
  // file is the audio file to pass to whisper for transcription
  if(is_readable($file)) {
      $mime = mime_content_type($file);
      if (class_exists('CURLFile')) {
          $attachment = new CURLFile($file, $mime, basename($file));
      } else {
          $attachment = '@' . $file . ';filename=' . $file . ';type=' . $mime;
      }
      $headers=array('Authorization: Bearer '.$token, 'Content-Type: multipart/form-data');
      $ch = curl_init('https://api.openai.com/v1/audio/transcriptions');
      curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
      curl_setopt($ch, CURLOPT_HEADER, 0);
      curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
      curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
      curl_setopt($ch, CURLOPT_TIMEOUT, 500);
      curl_setopt($ch, CURLOPT_VERBOSE, false);
      curl_setopt($ch, CURLOPT_SSLVERSION, 1);
      curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
      curl_setopt($ch, CURLOPT_POST, 1);
      curl_setopt($ch, CURLOPT_POSTFIELDS, array(
          'model' => 'whisper-1',
          'response_format' => 'text',
          'file' => $attachment
      ));
      $transcript = curl_exec($ch);
      curl_close($ch);
      $transcript = trim($transcript);
      return $transcript;
  } else {
      return "";
  }
}

function openai_tts($text,$voice,$token) {

    bot_debug($logfile,"tts $text");
    $text = removeEmojis($text);
    bot_debug($logfile,"tts2 $text");
    $url = 'https://api.openai.com/v1/audio/speech';
    $data = [
        'model' => 'tts-1',
        'input' => $text,
        'voice' => $voice,
        'response_format' => 'opus',
        'prompt' => "Esta es una respuesta en español"
    ];
    $headers = [
        'Authorization: Bearer ' . $token,
        'Content-Type: application/json',
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

    $response = curl_exec($ch);
    $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
    bot_debug($logfile,"content type $contentType");

    if($contentType=="") {
        return null;
    }

    $tempFile = tempnam(sys_get_temp_dir(), 'audio_');
    $tempFile .= ".ogg";

    if (curl_errno($ch)) {
        echo 'Error:' . curl_error($ch);
    } else {
        file_put_contents($tempFile, $response);
    }

    curl_close($ch);

    return array(
        'file' => $tempFile,
        'mime' => "audio/ogg",
        'filename' => "speech.ogg"
    );
}

function removeEmojis($text) {
    // Input validation
    if (!is_string($text)) {
        throw new InvalidArgumentException('Input must be a string');
    }

    // Return empty string if input is empty
    if (empty($text)) {
        return '';
    }

    // Pattern to match emoji characters
    $pattern = '/[\x{1F000}-\x{1F9FF}]'  // Miscellaneous Symbols and Pictographs
             . '|[\x{2600}-\x{26FF}]'    // Miscellaneous Symbols
             . '|[\x{2700}-\x{27BF}]'    // Dingbats
             . '|[\x{2300}-\x{23FF}]'    // Miscellaneous Technical
             . '|[\x{2B50}]'             // White Medium Star
             . '|[\x{2934}-\x{2935}]'    // Arrow characters
             . '|[\x{2B05}-\x{2B07}]'    // Directional arrows
             . '|[\x{2B1B}-\x{2B1C}]'    // Black and white squares
             . '|[\x{3030}]'             // Wavy dash
             . '|[\x{303D}]'             // Part alternation mark
             . '|[\x{3297}]'             // Circled ideograph congratulation
             . '|[\x{3299}]'             // Circled ideograph secret
             . '|[\x{FE0F}]'             // Variation selector
             . '|[\x{1F600}-\x{1F64F}]'  // Emoticons
             . '|[\x{1F300}-\x{1F5FF}]'  // Misc Symbols and Pictographs
             . '|[\x{1F680}-\x{1F6FF}]'  // Transport and Map
             . '|[\x{1F900}-\x{1F9FF}]'  // Supplemental Symbols and Pictographs
             . '|[\x{1F1E0}-\x{1F1FF}]'  // Flags (iOS)
             . '/u';                      // UTF-8 flag

    // Remove emojis using preg_replace
    $cleanText = preg_replace($pattern, '', $text);
    
    // Remove any zero-width joiner characters that might be left
    $cleanText = preg_replace('/[\x{200D}]/u', '', $cleanText);
    
    return $cleanText;
}


