<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
require_once __DIR__ . '/BotInterface.php';

class OpenAIAssistantBot implements BotInterface {

    public static function getKey() {
        return 'openai_assistant';
    }
    
    public static function getName() {
        return 'OpenAI Assistant';
    }
    
    public static function getTemplate() {
        return array(
            "openai_token"      => array( "type" => "text",     "default" => ""),
            "assistant_id"      => array( "type" => "text",     "default" => ""),
            "guardrails_prompt" => array( "type" => "textarea", "default" => ". Do not answer to anything that is outside your retrieval context."),
            "debug"             => array( "type" => "select",   "default" => "no", "options"=>array("yes"=>__('Yes'),"no"=>__('No')))
        );
    }
}

if (!defined('BOT_REGISTRATION_MODE') || !BOT_REGISTRATION_MODE) {
    require_once __DIR__ . '/common_functions.php';

    $iniFile = isset($_REQUEST['bot']) ? getcwd()."/".$_REQUEST['bot'].".ini" : null;
    file_put_contents("/tmp/borrame_nico.log","ini $iniFile\n",FILE_APPEND);
    $config = BotConfig::load('OpenAIAssistantBot', $iniFile);
    $logfile = basename(__FILE__,'.php');

    file_put_contents("/tmp/borrame_nico.log",print_r($config,1),FILE_APPEND);

    if ($config === null) {
        header('HTTP/1.0 403 Forbidden');
        die("Configuration file missing or inaccessible");
    }

    // Get bot input DATA
    $json = file_get_contents('php://input');
    $data = json_decode($json,1);

    if(!isset($data)) { 
        header('HTTP/1.0 403 Forbidden');
        die("Bad call");
    }
    bot_debug($logfile,"POST RECEIVED:\n".print_r($data,1));

    if($config['openai_token']=='' || $config['assistant_id']=='') {
        $response = "Bot is not yet configured, be sure to set up your OpenAI API key";
        send_response($response); // dies after sending response
    }

    $tools = new stdClass();
    $dorequest=1;
    $filename_tools = getcwd()."/".$_REQUEST['bot']."_tools.json";
    if(is_readable($filename_tools)) {
        $last = filemtime($filename_tools);
        $now = time();
        $dif = $now-$last;
        if($dif>360) {
            $dorequest=1;
        } else {
            $dorequest=0;
        }
    } else {
        $dorequest=1;
    }

    if($dorequest==1) {
        $tools = openai_get_assistant_tools($config['openai_token'],$config['assistant_id']);
        file_put_contents($filename_tools,json_encode($tools));
    } else {
        $tools = json_decode(file_get_contents($filename_tools));
    }

    bot_debug($logfile,print_r($tools,1));

    $current_state = '';

    // For assistants, the bot state is the openai thread
    if($data['state']=='') {
        // No state, its a new conversation, create a thread
        $thread = openai_create_thread($config['openai_token'],$tools);
        $current_state = array("thread"=>$thread);
    } else {
        $bot_state = json_decode($data['state'],1);
        $thread = $bot_state['thread'];
    }

    if($thread=='' || $thread === null) {
        $response = "Digital brain is not working correctly (A01) [transfer]";
        send_response($response); // dies after sending response
    }

    $user_prompt = $data['body'];
    $user_prompt.=". ".$config['guardrails_prompt']; // add guardrails to question

    $msgid = openai_create_message($config['openai_token'],$thread,$user_prompt);

    if($msgid=='' || $msgid === null) {
        $response = "Digital brain is not working correctly (A02) [transfer]";
        send_response($response); // dies after sending response
    }

    $runid = openai_create_run($config['openai_token'],$config['assistant_id'],$thread);
    if($runid=='' || $runid === null) {
        $response = "Digital brain is not working correctly (A03) [transfer]";
        send_response($response); // dies after sending response
    }

    // Get run status up to 30 attempts/seconds
    $ok=0;
    $inc=1000000;
    for($a=1;$a<60;$a++) {
        if ($a % 10 == 0) {
            $inc = (1000000 + ($a / 10) * 500000);
        }
        usleep($inc);
        $data = openai_retrieve_run($config['openai_token'],$thread,$runid);
        $status = $data['status'];
        bot_debug($logfile,"get run status $status");
        if($status=='completed') { 
            $ok=1; 
            break; 
        } else if($status=='requires_action') {
            $return = call_functions($config['openai_token'],$thread,$runid,$data['required_action']['submit_tool_outputs']['tool_calls']);
        } 
    }

    if($ok==0) {
        $response = "Digital brain is not working correctly (A04) [transfer]";
        send_response($response); // dies after sending response
    }

    // Retrieve first one from list (desc order limit 2)
    $messages = openai_list_messages($config['openai_token'],$thread);
    $openai_assistant_reply =  $messages[0]['content'][0]['text']['value'];

    $command = extract_command_from_reply($openai_assistant_reply);
    if($command=='close' || $command=='transfer') {
        openai_delete_thread($config['openai_token'],$thread);
        bot_debug($logfile,"OpenAI delete thread $thread");
    }

    $final_msg = remove_annotations($openai_assistant_reply);
    send_response($final_msg);
    // END OF PROGRAM
    exit;
}

function openai_create_thread($openai_token,$tools) {
    global $logfile;
    $url = 'https://api.openai.com/v1/threads';
    $final_tools = array("tool_resources"=>$tools);
    $data_string = json_encode($final_tools);
    $headers=array('Content-Type: application/json','Content-Length: ' . strlen($data_string),"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"create thread response: ".print_r($response,1));
    return $response['id'];
}

function openai_delete_thread($openai_token,$threadid) {
    global $logfile;
    $url = 'https://api.openai.com/v1/threads/'.$threadid;
    $headers=array('Content-Type: application/json',"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "DELETE");
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"delete trhead response: ".print_r($response,1));
    return $response['deleted'];
}

function openai_create_message($openai_token,$threadid,$content) {
    global $logfile;
    $url = 'https://api.openai.com/v1/threads/'.$threadid."/messages";
    $data = array(
       "role"=> "user",
       "content" => $content
    );
    $data_string = json_encode($data);
    $headers=array('Content-Type: application/json','Content-Length: ' . strlen($data_string),"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"create message url: ".$url);
    bot_debug($logfile,"create message payload: ". $data_string);
    bot_debug($logfile,"create message response: ".print_r($response,1));
    return $response['id'];
}

function openai_list_messages($openai_token,$threadid) {
    global $logfile;
    $url = 'https://api.openai.com/v1/threads/'.$threadid."/messages?order=desc&limit=2";
    $headers=array('Content-Type: application/json',"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"create message response: ".print_r($response,1));
    return $response['data'];
}

function openai_create_run($openai_token,$assistantid,$threadid) {
    global $logfile;
    $url = 'https://api.openai.com/v1/threads/'.$threadid."/runs";
    $data = array(
       "assistant_id"=> $assistantid
    );
    $data_string = json_encode($data);
    $headers=array('Content-Type: application/json','Content-Length: ' . strlen($data_string),"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, 1);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"create run response: ".print_r($response['status'],1));
    return $response['id'];
}

function openai_retrieve_run($openai_token,$threadid,$runid) {
    global $logfile;
    $url = 'https://api.openai.com/v1/threads/'.$threadid."/runs/".$runid;
    $headers=array('Content-Type: application/json',"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"retrieve run response: ".print_r($response['status'],1));
    return $response;
}

function openai_get_assistant_tools($openai_token,$assistant_id) {
    global $logfile;
    $url = 'https://api.openai.com/v1/assistants/'.$assistant_id;
    $headers=array('Content-Type: application/json',"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return);
    bot_debug($logfile,"get assistant return: ".print_r($response,1));

    if (isset($response->tool_resources)) {
        return $response->tool_resources;
    } else {
        bot_debug($logfile, "tool_resources not found in response");
        return new stdClass();
    }
}

function openai_submit_tool_output($openai_token,$threadid,$runid,$content) {
    global $logfile;
    $url = 'https://api.openai.com/v1/threads/'.$threadid."/runs/".$runid."/submit_tool_outputs";
    // Process tool_outputs to ensure output fields are strings
    $processed_content = array();
    foreach ($content as $tool_output) {
        $processed_output = $tool_output;
        if (isset($tool_output['output']) && (is_array($tool_output['output']) || is_object($tool_output['output']))) {
            $processed_output['output'] = json_encode($tool_output['output']);
        }
        $processed_content[] = $processed_output;
    }
    
    $data = array(
        'tool_outputs' => $processed_content
    );
    $data_string = json_encode($data);
    $headers=array('Content-Type: application/json','Content-Length: ' . strlen($data_string),"Authorization: Bearer $openai_token","OpenAI-Beta: assistants=v2");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"submit tool output url: ".$url);
    bot_debug($logfile,"submit tool output payload: ". $data_string);
    bot_debug($logfile,"submit tool output response: ".print_r($response,1));
    return $response['id'];
}

function call_functions($token,$thread,$runid,$tools) {
    global $config, $logfile;
    $func=array();
    $funcr = json_decode(base64_decode($FUNCTIONS),1);
    foreach($funcr as $idx=>$funcdata) {
       $func[$funcdata['name']]=$funcdata['url'];
    }

    $final_response = array();

    foreach($tools as $idx=>$data) {
        $tool_call_id = $data['id'];
        $func_name    = $data['function']['name']; 
        $func_args    = $data['function']['arguments'];
        if(isset($func[$func_name])) {
            bot_debug($logfile,"call function $func_name with args ".$func_args);
            $output = post_json_request($func[$func_name],$func_args);
            bot_debug($logfile,"response: $output");
        } else {
            bot_debug($logfile,"requested function $func_name is not defined in bot configuration");
            $output = array("success"=>"false");
        }
        $response = array("tool_call_id"=>$tool_call_id,"output"=>$output);
        $final_response[]=$response;
    }
    bot_debug($logfile,"about to submit output");
    bot_debug($logfile,print_r($final_response,1));
    openai_submit_tool_output($token,$thread,$runid,$final_response);
}

function post_json_request($url,$payload) {
    global $logfile;
    bot_debug($logfile,"do post to $url");
    $ch = curl_init( $url );
    curl_setopt( $ch, CURLOPT_POSTFIELDS, $payload );
    curl_setopt( $ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
    curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    $result = curl_exec($ch);
    bot_debug($logfile,"result: $result");
    curl_close($ch);
    return $result;
}

function extract_command_from_reply($text) {
    $pattern = '/\[(.*?)\]/';
    preg_match($pattern, $text, $matches);
    $extractedString = (isset($matches[1])) ? $matches[1] : '';
    return $extractedString;
}

function remove_annotations($text) {
    $pattern = '/【(.*?)】/';
    $replacedString = preg_replace($pattern, '', $text);
    return $replacedString;
}

function send_response($response) {
    global $current_state, $logfile;
    $return   = array("body"=>$response,"attach"=>'');
    if($current_state!='') {
        $return['state']=$current_state;
    }
    bot_debug($logfile,"REPLY: ".print_r($return,1));
    echo json_encode($return);
    exit;
}

