<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
require_once __DIR__ . '/BotInterface.php';

class ClaudeBot implements BotInterface {
    public static function getKey() {
        return 'claude';
    }
    
    public static function getName() {
        return 'Claude';
    }
    
    public static function getTemplate() {

        $models = array(
            "claude-3-haiku-20240307"=>"claude-3-haiku-20240307",
            "claude-3-sonnet-20240229"=>"claude-3-sonnet-20240229",
            "claude-3-opus-20240229"=>"claude-3-opus-20240229",
            "claude-3-5-sonnet-20240620"=>"claude-3-5-sonnet-20240620"
        );

        return array(
            "anthropic_token"   => array( "type" => "text",     "default" => ""),
            "anthropic_version" => array( "type" => "text",     "default" => "2023-06-01"),
            "max_tokens"        => array( "type" => "number",     "default" => "1024"),
            "model"             => array( "type" => "select",   "default" => "claude-3-haiku-20240307", "options"=>$models),
            "system_prompt"     => array( "type" => "textarea", "default" => "You are a helpful assistant"),
            "action_prompt"     => array( "type" => "textarea", "default" => "If the user is frustrated or angry, or if the user asks to connect to a real person, you will pass the conversation to a human by appending '[transfer]' to your reply. In the farewell/goodbye message you will append '[close]' to your reply."),
            "guardrails_prompt" => array( "type" => "textarea", "default"=>". Do not answer anything that is not in your system prompt context."),
            "debug"             => array( "type" => "select", "options"=>array("yes"=>__('Yes'),"no"=>__('No')), "default"=>"No")
        );
    }
}


if (!defined('BOT_REGISTRATION_MODE') || !BOT_REGISTRATION_MODE) {
    require_once __DIR__ . '/common_functions.php';

    $iniFile = isset($_REQUEST['bot']) ? getcwd()."/".$_REQUEST['bot'].".ini" : null;
    $config = BotConfig::load('ClaudeBot', $iniFile);
    $logfile = basename(__FILE__,'.php');

    if ($config === null) {
        header('HTTP/1.0 403 Forbidden');
        die("Configuration file missing or inaccessible");
    }

    // Get bot input DATA
    $json = file_get_contents('php://input');
    $data = json_decode($json,1);

    if(!isset($data)) { 
        header('HTTP/1.0 403 Forbidden');
        die("Bad call");
    }
    bot_debug($logfile,"POST RECEIVED:\n".print_r($data,1));

    $user_prompt = $data['body'];

    $user_prompt.=$config['guardrails_prompt']; // add guardrails to question

    $chat_context = array_slice($data['chat_context'],-6);

    if (count($chat_context) > 0 && $chat_context[0]['role'] === 'assistant') {
        array_shift($chat_context);
    }

    $chat_context[] = array("role"=>"user","content"=>$user_prompt);

    if($config['anthropic_token']=='') {
        $response = "Bot is not yet configured, be sure to set up your Claude API key";
    } else {
        $response = ask_claude($config['anthropic_token'],$config['system_prompt']." ".$config['action_prompt'],$chat_context);
    }

    $return = array("body"=>$response,"attach"=>'');
    bot_debug($logfile,"REPLY: $response");
    echo json_encode($return);
    die();
}

function ask_claude($anthropic_token,$system_prompt,$user_prompt) {
    global $config, $logfile;

    $url = 'https://api.anthropic.com/v1/messages';
    $data = array(
        "model"=> $config['model'],
        "max_tokens"=> intval($config['max_tokens']),
        "messages" => $user_prompt,
        "system" => $system_prompt,
    );
    $data_string = json_encode($data);
    bot_debug($logfile,"Claude Data: ".print_r($data,1));
    bot_debug($logfile,"Claude Messages: ".print_r($user_prompt,1));
    $headers=array('Content-Type: application/json','Content-Length: ' . strlen($data_string),"x-api-key: $anthropic_token","anthropic-version: {$config['anthropic_version']}");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 120);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"Claude response: ".print_r($response,1));
    return $response['content'][0]['text'];
}
