# FOP2 Chat Widget

A customizable chat widget that integrates with FOP2 (Flash Operator Panel 2) systems, providing real-time chat functionality with browser fingerprinting for user identification.

## Installation

Include the chat widget script in your HTML:

```html
<script src="path/to/chat-widget.js"></script>
```

## Setup Configuration

The `fop2chatbroker.Setup()` function accepts an object with various configuration options:

### Basic Configuration

#### `page` (Required)
The page identifier used for distinct webWidget sessions, if you configure more than one.

```javascript
fop2chatbroker.Setup({
    page: "some_name_or_page"
});
```

#### `lang` (Optional)
Language code for localization. Supported languages: `en` (English), `es` (Spanish), `pt_BR` (Portuguese).

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    lang: "es" // Spanish
});
```

#### `theme` (Optional)
Theme selection for the chat widget. Available themes: `default`, `whatsapp`, `modern`, `dark`.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    theme: "whatsapp" // WhatsApp-like theme
});
```

### Appearance Customization

#### `bubble_color` (Optional)
Background color for the chat bubble and header.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    bubble_color: "#2563eb" // Blue color
});
```

#### `text_color` (Optional)
Text color for the chat interface elements.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    text_color: "#ffffff" // White text
});
```

#### `avatar` (Optional)
URL to an avatar image for the agent/bot.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    avatar: "/images/agent-avatar.png"
});
```

### User Management

#### `autocreate_user` (Optional)
Automatically create user accounts without showing an initial FORM asking for data like name, email, etc

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    autocreate_user: true
});
```

#### `form_fields` (Optional)
Dynamic form fields for user registration if autocreate_user is set to false. Accepts a JSON string with field definitions. "name" is mandatory.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    form_fields: JSON.stringify([
        {
            fieldname: "name",
            fieldlabel: "Your Name",
            fieldtype: "text"
        },
        {
            fieldname: "email", 
            fieldlabel: "Email Address",
            fieldtype: "email"
        },
        {
            fieldname: "department",
            fieldlabel: "Department",
            fieldtype: "select",
            options: ["Sales", "Support", "Billing"]
        }
    ])
});
```

### Fingerprint Configuration

#### `fingerprint_identity` (Optional)
Use a predefined static identity instead of browser fingerprinting. This is the most secure option when you already know the user's identity.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    fingerprint_identity: "user-12345" // Static user ID
});
```

#### `fingerprint_hasher` (Optional)
Custom hash function for fingerprint generation. Useful when you want to incorporate known user identities into the fingerprint.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    fingerprint_hasher: function(keys, seed) {
        // Example: Combine known user ID with browser fingerprint
        const userIdentity = "known-user-id-123";
        const combined = userIdentity + keys + seed;
        
        // Simple hash function
        let hash = 0;
        for (let i = 0; i < combined.length; i++) {
            const char = combined.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash; // Convert to 32bit integer
        }
        return hash;
    }
});
```

**Security Note**: The custom hasher function is wrapped in error handling. If it throws an error, the system falls back to the default murmurhash algorithm.

### Chat Behavior

#### `greeting` (Optional)
Custom greeting message with variable support using `{{fieldname}}` syntax.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    greeting: "Hello {{name}}, welcome to our support chat!"
});
```

#### `force_open` (Optional)
Automatically open the chat popup after initialization.

```javascript
fop2chatbroker.Setup({
    page: "example.com",
    force_open: true
});
```

## Complete Example

```javascript
fop2chatbroker.Setup({
    page: "example.com/contact-us",
    lang: "en",
    theme: "modern",
    bubble_color: "#007bff",
    text_color: "#ffffff",
    avatar: "/images/support-agent.png",
    autocreate_user: false,
    form_fields: JSON.stringify([
        {
            fieldname: "full_name",
            fieldlabel: "Full Name",
            fieldtype: "text"
        },
        {
            fieldname: "email",
            fieldlabel: "Email",
            fieldtype: "email"
        }
    ]),
    fingerprint_identity: "customer-john-doe-456",
    greeting: "Hello {{full_name}}, how can we help you today?",
    force_open: false
});
```

## API Methods

### `fop2chatbroker.setTheme(themeName)`
Change the theme dynamically.

```javascript
fop2chatbroker.setTheme("dark");
```

### `fop2chatbroker.getAvailableThemes()`
Get list of available themes.

```javascript
const themes = fop2chatbroker.getAvailableThemes();
console.log(themes); // ["default", "whatsapp", "modern", "dark"]
```

### `fop2chatbroker.sendReply(element)`
Programmatically send a reply from button elements.

## Fingerprint Security Considerations

### When to use `fingerprint_identity`:
- When you already know the user's identity (logged-in users)
- For consistent user identification across sessions
- When browser fingerprinting privacy is a concern

### When to use `fingerprint_hasher`:
- When you want to combine browser fingerprint with known user data
- For custom fingerprint generation logic
- When you need specific hash algorithm requirements

### Default Behavior:
- Uses browser fingerprinting with murmurhash3 algorithm
- Collects: userAgent, language, screen resolution, timezone, storage capabilities, plugins, canvas fingerprint
- Provides unique identification for anonymous users

## Error Handling

The chat widget includes comprehensive error handling:
- Invalid custom hasher functions fall back to default algorithm
- Network errors are gracefully handled
- Malformed configuration options are ignored with warnings

## Browser Support

- Chrome 60+
- Firefox 55+
- Safari 12+
- Edge 79+

## License

This chat widget is part of the FOP2 system. Please refer to your FOP2 licensing agreement for usage terms.
